/*****************************************************************************\
*                                                                             *
* shellapi.h -  SHELL.DLL functions, types, and definitions                   *
*                                                                             *
* Copyright (c) Microsoft Corporation. All rights reserved.                   *
*                                                                             *
\*****************************************************************************/

#ifndef _INC_SHELLAPI
#define _INC_SHELLAPI

#include <SpecStrings.h>

//
// Define API decoration for direct importing of DLL references.
//
#ifndef WINSHELLAPI
#if !defined(_SHELL32_)
#define WINSHELLAPI       DECLSPEC_IMPORT
#else
#define WINSHELLAPI
#endif
#endif // WINSHELLAPI

#ifndef SHSTDAPI
#if !defined(_SHELL32_)
#define SHSTDAPI          EXTERN_C DECLSPEC_IMPORT HRESULT STDAPICALLTYPE
#define SHSTDAPI_(type)   EXTERN_C DECLSPEC_IMPORT type STDAPICALLTYPE
#else
#define SHSTDAPI          STDAPI
#define SHSTDAPI_(type)   STDAPI_(type)
#endif
#endif // SHSTDAPI

#ifndef SHDOCAPI
#if !defined(_SHDOCVW_)
#define SHDOCAPI          EXTERN_C DECLSPEC_IMPORT HRESULT STDAPICALLTYPE
#define SHDOCAPI_(type)   EXTERN_C DECLSPEC_IMPORT type STDAPICALLTYPE
#else
#define SHDOCAPI          STDAPI
#define SHDOCAPI_(type)   STDAPI_(type)
#endif
#endif // SHDOCAPI


#if !defined(_WIN64)
#include <pshpack1.h>
#endif

#ifdef __cplusplus
extern "C" {            /* Assume C declarations for C++ */
#endif  /* __cplusplus */



DECLARE_HANDLE(HDROP);

SHSTDAPI_(UINT) DragQueryFileA(HDROP hDrop, UINT iFile, __out_ecount_opt(cch) LPSTR lpszFile, UINT cch);
SHSTDAPI_(UINT) DragQueryFileW(HDROP hDrop, UINT iFile, __out_ecount_opt(cch) LPWSTR lpszFile, UINT cch);
#ifdef UNICODE
#define DragQueryFile  DragQueryFileW
#else
#define DragQueryFile  DragQueryFileA
#endif // !UNICODE
SHSTDAPI_(BOOL) DragQueryPoint(HDROP hDrop,__out LPPOINT lppt);
SHSTDAPI_(void) DragFinish(__in HDROP hDrop);
SHSTDAPI_(void) DragAcceptFiles(__in HWND hWnd, __in BOOL fAccept);

SHSTDAPI_(HINSTANCE) ShellExecuteA(HWND hwnd, LPCSTR lpOperation, LPCSTR lpFile, LPCSTR lpParameters,
    LPCSTR lpDirectory, INT nShowCmd);
SHSTDAPI_(HINSTANCE) ShellExecuteW(HWND hwnd, LPCWSTR lpOperation, LPCWSTR lpFile, LPCWSTR lpParameters,
    LPCWSTR lpDirectory, INT nShowCmd);
#ifdef UNICODE
#define ShellExecute  ShellExecuteW
#else
#define ShellExecute  ShellExecuteA
#endif // !UNICODE
SHSTDAPI_(HINSTANCE) FindExecutableA(LPCSTR lpFile, LPCSTR lpDirectory, __out_ecount(MAX_PATH) LPSTR lpResult);
SHSTDAPI_(HINSTANCE) FindExecutableW(LPCWSTR lpFile, LPCWSTR lpDirectory, __out_ecount(MAX_PATH) LPWSTR lpResult);
#ifdef UNICODE
#define FindExecutable  FindExecutableW
#else
#define FindExecutable  FindExecutableA
#endif // !UNICODE
SHSTDAPI_(LPWSTR *)  CommandLineToArgvW(LPCWSTR lpCmdLine, __out int* pNumArgs);

SHSTDAPI_(INT) ShellAboutA(HWND hWnd, LPCSTR szApp, LPCSTR szOtherStuff, HICON hIcon);
SHSTDAPI_(INT) ShellAboutW(HWND hWnd, LPCWSTR szApp, LPCWSTR szOtherStuff, HICON hIcon);
#ifdef UNICODE
#define ShellAbout  ShellAboutW
#else
#define ShellAbout  ShellAboutA
#endif // !UNICODE
SHSTDAPI_(HICON) DuplicateIcon(HINSTANCE hInst, HICON hIcon);
SHSTDAPI_(HICON) ExtractAssociatedIconA(HINSTANCE hInst, __inout_ecount(128)  LPSTR lpIconPath, __inout LPWORD lpiIcon);
SHSTDAPI_(HICON) ExtractAssociatedIconW(HINSTANCE hInst, __inout_ecount(128)  LPWSTR lpIconPath, __inout LPWORD lpiIcon);
#ifdef UNICODE
#define ExtractAssociatedIcon  ExtractAssociatedIconW
#else
#define ExtractAssociatedIcon  ExtractAssociatedIconA
#endif // !UNICODE
SHSTDAPI_(HICON) ExtractAssociatedIconExA(HINSTANCE hInst, __inout_ecount(128) LPSTR lpIconPath, __inout LPWORD lpiIconIndex, __out LPWORD lpiIconId);
SHSTDAPI_(HICON) ExtractAssociatedIconExW(HINSTANCE hInst, __inout_ecount(128) LPWSTR lpIconPath, __inout LPWORD lpiIconIndex, __out LPWORD lpiIconId);
#ifdef UNICODE
#define ExtractAssociatedIconEx  ExtractAssociatedIconExW
#else
#define ExtractAssociatedIconEx  ExtractAssociatedIconExA
#endif // !UNICODE
SHSTDAPI_(HICON) ExtractIconA(HINSTANCE hInst, LPCSTR lpszExeFileName, UINT nIconIndex);
SHSTDAPI_(HICON) ExtractIconW(HINSTANCE hInst, LPCWSTR lpszExeFileName, UINT nIconIndex);
#ifdef UNICODE
#define ExtractIcon  ExtractIconW
#else
#define ExtractIcon  ExtractIconA
#endif // !UNICODE


#if(WINVER >= 0x0400)
typedef struct _DRAGINFOA {
    UINT uSize;                 /* init with sizeof(DRAGINFO) */
    POINT pt;
    BOOL fNC;
    LPSTR   lpFileList;
    DWORD grfKeyState;
} DRAGINFOA, *LPDRAGINFOA;
typedef struct _DRAGINFOW {
    UINT uSize;                 /* init with sizeof(DRAGINFO) */
    POINT pt;
    BOOL fNC;
    LPWSTR  lpFileList;
    DWORD grfKeyState;
} DRAGINFOW, *LPDRAGINFOW;
#ifdef UNICODE
typedef DRAGINFOW DRAGINFO;
typedef LPDRAGINFOW LPDRAGINFO;
#else
typedef DRAGINFOA DRAGINFO;
typedef LPDRAGINFOA LPDRAGINFO;
#endif // UNICODE


////
//// AppBar stuff
////
#define ABM_NEW           0x00000000
#define ABM_REMOVE        0x00000001
#define ABM_QUERYPOS      0x00000002
#define ABM_SETPOS        0x00000003
#define ABM_GETSTATE      0x00000004
#define ABM_GETTASKBARPOS 0x00000005
#define ABM_ACTIVATE      0x00000006  // lParam == TRUE/FALSE means activate/deactivate
#define ABM_GETAUTOHIDEBAR 0x00000007
#define ABM_SETAUTOHIDEBAR 0x00000008  // this can fail at any time.  MUST check the result
                                        // lParam = TRUE/FALSE  Set/Unset
                                        // uEdge = what edge
#define ABM_WINDOWPOSCHANGED 0x0000009
#if (NTDDI_VERSION >= NTDDI_WINXP)
#define ABM_SETSTATE         0x0000000a
#endif // (NTDDI_VERSION >= NTDDI_WINXP)

// these are put in the wparam of callback messages
#define ABN_STATECHANGE    0x0000000
#define ABN_POSCHANGED     0x0000001
#define ABN_FULLSCREENAPP  0x0000002
#define ABN_WINDOWARRANGE  0x0000003 // lParam == TRUE means hide

// flags for get state
#define ABS_AUTOHIDE    0x0000001
#define ABS_ALWAYSONTOP 0x0000002

#define ABE_LEFT        0
#define ABE_TOP         1
#define ABE_RIGHT       2
#define ABE_BOTTOM      3

typedef struct _AppBarData
{
    DWORD cbSize;
    HWND hWnd;
    UINT uCallbackMessage;
    UINT uEdge;
    RECT rc;
    LPARAM lParam; // message specific
} APPBARDATA, *PAPPBARDATA;


SHSTDAPI_(UINT_PTR) SHAppBarMessage(DWORD dwMessage, __in PAPPBARDATA pData);

////
////  EndAppBar
////

SHSTDAPI_(DWORD)   DoEnvironmentSubstA(__inout_ecount(cchString) LPSTR szString, UINT cchString);
SHSTDAPI_(DWORD)   DoEnvironmentSubstW(__inout_ecount(cchString) LPWSTR szString, UINT cchString);
#ifdef UNICODE
#define DoEnvironmentSubst  DoEnvironmentSubstW
#else
#define DoEnvironmentSubst  DoEnvironmentSubstA
#endif // !UNICODE

#define EIRESID(x) (-1 * (int)(x))
SHSTDAPI_(UINT) ExtractIconExA(LPCSTR lpszFile, int nIconIndex, __out_ecount_opt(nIcons) HICON *phiconLarge, __out_ecount_opt(nIcons) HICON *phiconSmall, UINT nIcons);
SHSTDAPI_(UINT) ExtractIconExW(LPCWSTR lpszFile, int nIconIndex, __out_ecount_opt(nIcons) HICON *phiconLarge, __out_ecount_opt(nIcons) HICON *phiconSmall, UINT nIcons);
#ifdef UNICODE
#define ExtractIconEx  ExtractIconExW
#else
#define ExtractIconEx  ExtractIconExA
#endif // !UNICODE

// Shell File Operations

#define FO_MOVE                    0x0001
#define FO_COPY                    0x0002
#define FO_DELETE                  0x0003
#define FO_RENAME                  0x0004

// SHFILEOPSTRUCT.fFlags and IFileOperation::SetOperationFlags() flag values

#define FOF_MULTIDESTFILES         0x0001
#define FOF_CONFIRMMOUSE           0x0002
#define FOF_SILENT                 0x0004  // don't display progress UI (confirm prompts may be displayed still)
#define FOF_RENAMEONCOLLISION      0x0008  // automatically rename the source files to avoid the collisions
#define FOF_NOCONFIRMATION         0x0010  // don't display confirmation UI, assume "yes" for cases that can be bypassed, "no" for those that can not
#define FOF_WANTMAPPINGHANDLE      0x0020  // Fill in SHFILEOPSTRUCT.hNameMappings
                                           // Must be freed using SHFreeNameMappings
#define FOF_ALLOWUNDO              0x0040  // enable undo including Recycle behavior for IFileOperation::Delete()
#define FOF_FILESONLY              0x0080  // only operate on the files (non folders), both files and folders are assumed without this
#define FOF_SIMPLEPROGRESS         0x0100  // means don't show names of files
#define FOF_NOCONFIRMMKDIR         0x0200  // don't dispplay confirmatino UI before making any needed directories, assume "Yes" in these cases
#define FOF_NOERRORUI              0x0400  // don't put up error UI, other UI may be displayed, progress, confirmations
#if (_WIN32_IE >= 0x0500)
#define FOF_NOCOPYSECURITYATTRIBS  0x0800  // dont copy file security attributes (ACLs)
#define FOF_NORECURSION            0x1000  // don't recurse into directories for operations that would recurse
#define FOF_NO_CONNECTED_ELEMENTS  0x2000  // don't operate on connected elements ("xxx_files" folders that go with .htm files)
#define FOF_WANTNUKEWARNING        0x4000  // during delete operation, warn if nuking instead of recycling (partially overrides FOF_NOCONFIRMATION)
#endif // (_WIN32_IE >= 0x500)
#if (_WIN32_WINNT >= 0x0501)
#define FOF_NORECURSEREPARSE       0x8000  // deprecated; the operations engine always does the right thing on FolderLink objects (symlinks, reparse points, folder shortcuts)
#endif // (_WIN32_WINNT >= 0x501)
#define FOF_NO_UI                   (FOF_SILENT | FOF_NOCONFIRMATION | FOF_NOERRORUI | FOF_NOCONFIRMMKDIR) // don't display any UI at all

typedef WORD FILEOP_FLAGS;

#define PO_DELETE       0x0013  // printer is being deleted
#define PO_RENAME       0x0014  // printer is being renamed
#define PO_PORTCHANGE   0x0020  // port this printer connected to is being changed
                                // if this id is set, the strings received by
                                // the copyhook are a doubly-null terminated
                                // list of strings.  The first is the printer
                                // name and the second is the printer port.
#define PO_REN_PORT     0x0034  // PO_RENAME and PO_PORTCHANGE at same time.

// no POF_ flags currently defined

typedef WORD PRINTEROP_FLAGS;

// implicit parameters are:
//      if pFrom or pTo are unqualified names the current directories are
//      taken from the global current drive/directory settings managed
//      by Get/SetCurrentDrive/Directory
//
//      the global confirmation settings

typedef struct _SHFILEOPSTRUCTA
{
    HWND            hwnd;
    UINT            wFunc;
    LPCSTR          pFrom;
    LPCSTR          pTo;
    FILEOP_FLAGS    fFlags;
    BOOL            fAnyOperationsAborted;
    LPVOID          hNameMappings;
    LPCSTR          lpszProgressTitle; // only used if FOF_SIMPLEPROGRESS
} SHFILEOPSTRUCTA, *LPSHFILEOPSTRUCTA;
typedef struct _SHFILEOPSTRUCTW
{
    HWND            hwnd;
    UINT            wFunc;
    LPCWSTR         pFrom;
    LPCWSTR         pTo;
    FILEOP_FLAGS    fFlags;
    BOOL            fAnyOperationsAborted;
    LPVOID          hNameMappings;
    LPCWSTR         lpszProgressTitle; // only used if FOF_SIMPLEPROGRESS
} SHFILEOPSTRUCTW, *LPSHFILEOPSTRUCTW;
#ifdef UNICODE
typedef SHFILEOPSTRUCTW SHFILEOPSTRUCT;
typedef LPSHFILEOPSTRUCTW LPSHFILEOPSTRUCT;
#else
typedef SHFILEOPSTRUCTA SHFILEOPSTRUCT;
typedef LPSHFILEOPSTRUCTA LPSHFILEOPSTRUCT;
#endif // UNICODE

SHSTDAPI_(int) SHFileOperationA(__inout LPSHFILEOPSTRUCTA lpFileOp);
SHSTDAPI_(int) SHFileOperationW(__inout LPSHFILEOPSTRUCTW lpFileOp);
#ifdef UNICODE
#define SHFileOperation  SHFileOperationW
#else
#define SHFileOperation  SHFileOperationA
#endif // !UNICODE
SHSTDAPI_(void) SHFreeNameMappings(HANDLE hNameMappings);

typedef struct _SHNAMEMAPPINGA
{
    LPSTR   pszOldPath;
    LPSTR   pszNewPath;
    int   cchOldPath;
    int   cchNewPath;
} SHNAMEMAPPINGA, *LPSHNAMEMAPPINGA;
typedef struct _SHNAMEMAPPINGW
{
    LPWSTR  pszOldPath;
    LPWSTR  pszNewPath;
    int   cchOldPath;
    int   cchNewPath;
} SHNAMEMAPPINGW, *LPSHNAMEMAPPINGW;
#ifdef UNICODE
typedef SHNAMEMAPPINGW SHNAMEMAPPING;
typedef LPSHNAMEMAPPINGW LPSHNAMEMAPPING;
#else
typedef SHNAMEMAPPINGA SHNAMEMAPPING;
typedef LPSHNAMEMAPPINGA LPSHNAMEMAPPING;
#endif // UNICODE


////
//// End Shell File Operations
////

////
////  Begin ShellExecuteEx and family
////

/* ShellExecute() and ShellExecuteEx() error codes */

/* regular WinExec() codes */
#define SE_ERR_FNF              2       // file not found
#define SE_ERR_PNF              3       // path not found
#define SE_ERR_ACCESSDENIED     5       // access denied
#define SE_ERR_OOM              8       // out of memory
#define SE_ERR_DLLNOTFOUND              32

#endif /* WINVER >= 0x0400 */

/* error values for ShellExecute() beyond the regular WinExec() codes */
#define SE_ERR_SHARE                    26
#define SE_ERR_ASSOCINCOMPLETE          27
#define SE_ERR_DDETIMEOUT               28
#define SE_ERR_DDEFAIL                  29
#define SE_ERR_DDEBUSY                  30
#define SE_ERR_NOASSOC                  31

#if(WINVER >= 0x0400)

// Note CLASSKEY overrides CLASSNAME
#define SEE_MASK_CLASSNAME         0x00000001
#define SEE_MASK_CLASSKEY          0x00000003
// Note INVOKEIDLIST overrides IDLIST
#define SEE_MASK_IDLIST            0x00000004
#define SEE_MASK_INVOKEIDLIST      0x0000000c
#if (NTDDI_VERSION < NTDDI_LONGHORN)
#define SEE_MASK_ICON              0x00000010  //  not used
#endif // (NTDDI_VERSION < NTDDI_LONGHORN)
#define SEE_MASK_HOTKEY            0x00000020
#define SEE_MASK_NOCLOSEPROCESS    0x00000040
#define SEE_MASK_CONNECTNETDRV     0x00000080
#define SEE_MASK_NOASYNC           0x00000100
#define SEE_MASK_FLAG_DDEWAIT      SEE_MASK_NOASYNC // Use SEE_MASK_NOASYNC instead of SEE_MASK_FLAG_DDEWAIT as it more accuratly describes the behavior
#define SEE_MASK_DOENVSUBST        0x00000200
#define SEE_MASK_FLAG_NO_UI        0x00000400
#define SEE_MASK_UNICODE           0x00004000
#define SEE_MASK_NO_CONSOLE        0x00008000
#define SEE_MASK_ASYNCOK           0x00100000
#if (NTDDI_VERSION >= NTDDI_WIN2K)
#define SEE_MASK_HMONITOR          0x00200000
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)
#if (NTDDI_VERSION >= NTDDI_WINXPSP1)
#define SEE_MASK_NOZONECHECKS      0x00800000
#endif // (NTDDI_VERSION >= NTDDI_WINXPSP1)
#if (NTDDI_VERSION >= NTDDI_WIN2K)
#define SEE_MASK_NOQUERYCLASSSTORE 0x01000000
#define SEE_MASK_WAITFORINPUTIDLE  0x02000000
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)
#if (NTDDI_VERSION >= NTDDI_WINXP)
#define SEE_MASK_FLAG_LOG_USAGE    0x04000000
#endif // (NTDDI_VERSION >= NTDDI_WINXP)



//
// For compilers that don't support nameless unions
//
#ifndef DUMMYUNIONNAME
#ifdef NONAMELESSUNION
#define DUMMYUNIONNAME   u
#define DUMMYUNIONNAME2  u2
#define DUMMYUNIONNAME3  u3
#define DUMMYUNIONNAME4  u4
#define DUMMYUNIONNAME5  u5
#else
#define DUMMYUNIONNAME
#define DUMMYUNIONNAME2
#define DUMMYUNIONNAME3
#define DUMMYUNIONNAME4
#define DUMMYUNIONNAME5
#endif
#endif // DUMMYUNIONNAME

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
#pragma warning(push)
#pragma warning(disable:4201) /* nonstandard extension used : nameless struct/union */
#pragma once
#endif

typedef struct _SHELLEXECUTEINFOA
{
        DWORD cbSize;
        ULONG fMask;
        HWND hwnd;
        LPCSTR   lpVerb;
        LPCSTR   lpFile;
        LPCSTR   lpParameters;
        LPCSTR   lpDirectory;
        int nShow;
        HINSTANCE hInstApp;
        // Optional fields
        LPVOID lpIDList;
        LPCSTR   lpClass;
        HKEY hkeyClass;
        DWORD dwHotKey;
        union {
        HANDLE hIcon;
#if (NTDDI_VERSION >= NTDDI_WIN2K)
        HANDLE hMonitor;
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)
        } DUMMYUNIONNAME;
        HANDLE hProcess;
} SHELLEXECUTEINFOA, *LPSHELLEXECUTEINFOA;
typedef struct _SHELLEXECUTEINFOW
{
        DWORD cbSize;
        ULONG fMask;
        HWND hwnd;
        LPCWSTR  lpVerb;
        LPCWSTR  lpFile;
        LPCWSTR  lpParameters;
        LPCWSTR  lpDirectory;
        int nShow;
        HINSTANCE hInstApp;
        // Optional fields
        LPVOID lpIDList;
        LPCWSTR  lpClass;
        HKEY hkeyClass;
        DWORD dwHotKey;
        union {
        HANDLE hIcon;
#if (NTDDI_VERSION >= NTDDI_WIN2K)
        HANDLE hMonitor;
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)
        } DUMMYUNIONNAME;
        HANDLE hProcess;
} SHELLEXECUTEINFOW, *LPSHELLEXECUTEINFOW;
#ifdef UNICODE
typedef SHELLEXECUTEINFOW SHELLEXECUTEINFO;
typedef LPSHELLEXECUTEINFOW LPSHELLEXECUTEINFO;
#else
typedef SHELLEXECUTEINFOA SHELLEXECUTEINFO;
typedef LPSHELLEXECUTEINFOA LPSHELLEXECUTEINFO;
#endif // UNICODE

SHSTDAPI_(BOOL) ShellExecuteExA(__inout LPSHELLEXECUTEINFOA lpExecInfo);
SHSTDAPI_(BOOL) ShellExecuteExW(__inout LPSHELLEXECUTEINFOW lpExecInfo);
#ifdef UNICODE
#define ShellExecuteEx  ShellExecuteExW
#else
#define ShellExecuteEx  ShellExecuteExA
#endif // !UNICODE

#if (NTDDI_VERSION >= NTDDI_WIN2K)
//
//  SHCreateProcessAsUser()
typedef struct _SHCREATEPROCESSINFOW
{
        DWORD cbSize;
        ULONG fMask;
        HWND hwnd;
        LPCWSTR  pszFile;
        LPCWSTR  pszParameters;
        LPCWSTR  pszCurrentDirectory;
        HANDLE hUserToken;
        LPSECURITY_ATTRIBUTES lpProcessAttributes;
        LPSECURITY_ATTRIBUTES lpThreadAttributes;
        BOOL bInheritHandles;
        DWORD dwCreationFlags;
        LPSTARTUPINFOW lpStartupInfo;
        LPPROCESS_INFORMATION lpProcessInformation;
} SHCREATEPROCESSINFOW, *PSHCREATEPROCESSINFOW;

SHSTDAPI_(BOOL) SHCreateProcessAsUserW(__inout PSHCREATEPROCESSINFOW pscpi);
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)

#if (NTDDI_VERSION >= NTDDI_LONGHORN)

SHSTDAPI SHEvaluateSystemCommandTemplate(PCWSTR pszCmdTemplate, __out PWSTR *ppszApplication, __out_opt PWSTR *ppszCommandLine, __out_opt PWSTR *ppszParameters);
//
//  SHEvaluateSystemCommandTemplate()
//      *   enforces stricter validation before calling CreateProcess().  may also be
//              used before calling ShellExecute().
//      *   should be used when caller wants the deterministic behavior from a command template
//              regardless of execution context.  it ignores the current process state,
//              such as the %PATH%, GetCurrentDirectory(), and parent process directory.
//      *   should be used when the command is hardcoded.
//      *   is used by ShellExecute() when handling file associations from HKCR.
//      *   reduces CreateProcess() commandline exploits
//      *   is not designed for processing user input, and may generate unexpected failures.
//
//  INPUT:
//      pszCmdTemplate =    command line, this may or may not include parameters.
//                          if the parameters are substitution parameters then this API
//                          should be called before parameters have been replaced.
//                          (check the examples below to see sample supported inputs.)
//
//  OUTPUT on return: S_OK
//      ppszApplication =   verified path to the Application.  this should be passed as the lpApplication
//                          parameter to CreateProcess() or the lpFile parameter to ShellExecute().
//                          (allocated using CoTaskMemAlloc(), free with CoTaskMemFree())
//
//      ppszCommandLine =   OPTIONAL - use if planning to call CreateProcess().
//                          resulting command line template.  parameters should be replaced based on this template,
//                          and then passed as the lpCommandLine parameter to CreateProcess().
//                          it is guaranteed to be of a form that PathGetArgs() will always succeed correctly.
//                          (allocated using CoTaskMemAlloc(), free with CoTaskMemFree())
//
//      ppszParameters  =   OPTIONAL - use if planning to call ShellExecute().
//                          resulting parameter list template.  parameters should be replaced based on this template,
//                          and then passed as the lpParameters parameter to ShellExecute().
//                          NOTE: identical to PathGetArgs(*ppszCommandLine).
//                          (allocated using CoTaskMemAlloc(), free with CoTaskMemFree())
//
//  OUTPUT on return: FAILED()
//      all outputs will be NULL'ed on failure
//
//  NOTES:  the parsing logic to determine a valid Application path is non-trivial, although
//              the extension is not required and if missing will be completed
//              in the following standard order:  { .PIF, .COM, .EXE, .BAT, .CMD }
//
//      Relative Paths are System Paths - if the first token has no path qualifiers
//              then the token is first checked to see if a key of the same name has
//              been installed under HKLM\Software\Microsoft\Windows\CurrentVersion\App Paths.
//              if the key or default value does not exist, it is assumed to be a child
//              of the system directories.  the following directories will be searched
//              in order for the relative token: { CSIDL_SYSTEM, CSIDL_WINDOWS }
//
//      Prefer Quoted Paths - if the first token in pszCmdTemplate is quoted and appears
//              to be an absolute path then the token is the only possible result.
//
//      Limit Forms of Unquoted Paths - if the first token is unquoted and appears
//              to be an absolute path, then it is subject to more stringent limitations.
//              if the token is a substring of CSIDL_PROGRAM_FILES or does not
//              exist on the file system, then SHEvaluateSystemCommandTemplate() will
//              attempt to complete using a token delimited by the first space of the
//              last valid path segment (usually the file name).  if this token also doesnt exist,
//              then the next space will be used, etc.
//
//  USAGE:      used before calling into CreateProcess() or ShellExecute(), callers
//              would typically look like the following:
/* #if 0 // SAMPLE CODE
HRESULT MyCreateProcessPriv(PCWSTR pszCmd)
{
    PWSTR pszApp;
    PWSTR pszCmdLine;
    HRESULT hr = SHEvaluateSystemCommandTemplate(pszCmd, &pszApp, &pszCmdLine);
    if (SUCCEEDED(hr))
    {
        //  if this was a real template, maybe some kind of wnsprintf() first?
        PROCESS_INFORMATION pi;
        STARTUPINFO si = {0};
        si.cb = sizeof(startup);
        si.wShowWindow = SW_SHOWNORMAL;

        if (CreateProcess(pszApp, pszCmdLine, NULL, NULL, FALSE,
                 CREATE_DEFAULT_ERROR_MODE, NULL, NULL, &si, &pi))
        {
            //  we are good
            ASSERT(hr == S_OK);
            CloseHandle(pi.hProcess);
            CloseHandle(pi.hThread);
        }
        else
        {
            hr = HRESULT_FROM_WIN32(GetLastError());
        }

        CoTaskMemFree(pszApp);
        CoTaskMemFree(pszCmdLine);
    }
    return hr;
}

HRESULT MyShellExec(PCWSTR pszCmd)
{
    PWSTR pszApp;
    PWSTR pszCmdLine;
    HRESULT hr = SHEvaluateSystemCommandTemplate(pszCmd, &pszApp, &pszCmdLine);
    if (SUCCEEDED(hr))
    {
        //  if this was a real template, maybe some kind of wnsprintf() first?
        SHELLEXECUTEINFOW sei = {
            sizeof(sei),           // cbSize;
            0,                     // fMask
            NULL,                  // hwnd
            NULL,                  // lpVerb
            pszApp,                // lpFile
            PathGetArgs(pszCmdLine), // lpParameters
            NULL,                  // lpDirectory
            SW_SHOWNORMAL,         // nShow
            0,                     // hInstApp
            NULL,                  // lpIDList
            NULL,                  // lpClass
            NULL,                  // hkeyClass
            0,                     // dwHotKey
            NULL,                  // hIcon
            NULL                   // hProcess
        };

        if (ShellExecuteEx(&sei))
        {
            //  we are good
            ASSERT(hr == S_OK);
        }
        else
        {
            hr = HRESULT_FROM_WIN32(GetLastError());
        }

        CoTaskMemFree(pszApp);
        CoTaskMemFree(pszCmdLine);
    }
    return hr;
}
#endif //  0 // SAMPLE CODE
*/

//  EXAMPLE:   Each example will show an input parameter and the results returned by
//              SHEvaluateSystemCommandTemplate().  Also included is the alternate result
//              of what CreateProcess() would have created if pszCmdTemplate were
//              passed directly as lpCommandLine and lpApplication were NULL.
//              (results marked with an asterisk (*) indicate differences.)
//
//          Assume for the examples that the following paths and values exist:
//
//      SHGetFolderPath() values:
//          CSIDL_SYSTEM            =   C:\windows\system32
//          CSIDL_WINDOWS           =   C:\windows
//          CSIDL_PROGRAM_FILES     =   C:\Program Files
//
//      Environment settings
//          GetModuleFileName(NULL) =   C:\Program Files\Example\sample.exe
//          GetCurrentDirectory()   =   \\server\share\foo
//          HKLM\...\App Paths\pbrush.exe = C:\windows\system32\mspaint.exe
//          HKLM\...\App Paths\mycl.exe = C:\Program Files\Compilers\mycl.exe
//          PATH                    =   c:\windows\system32;C:\windows;c:\;C:\Program Files\Compilers\
//
//      Valid Application paths:
//          C:\Program Files\Internet Explorer\iexplore.exe
//          C:\windows\system32\rundll32.exe
//          C:\windows\system32\notepad.exe
//          C:\windows\notepad.exe
//          C:\Program Files\Example\sample.exe
//          C:\Program Files\Compilers\cl.exe
//          C:\Other Programs\prog.exe
//
//      Suspicious (possibly hostile) Application paths:
//          C:\Program.exe
//          C:\Program Files\Internet.exe
//          C:\Program Files\Example\regedit.bat
//          C:\mycl.exe
//          \\server\share\foo\rundll32.exe
//          \\server\share\foo\myapp.exe
//
//
//  Relative Path Example #1
//      pszCmdTemplate      =   notepad.exe %1
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\windows\system32\notepad.exe
//              pszCommandLine  =   "notepad.exe" %1
//          CreateProcess() would return TRUE
//              new process =   C:\windows\system32\notepad.exe
//
//  Relative Path Example #2
//      pszCmdTemplate      =   rundll32.exe shell32.dll,RunDll
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\windows\system32\rundll32.exe
//              pszCommandLine  =   "rundll32.exe" shell32.dll,RunDll
//          * CreateProcess() would return TRUE
//              new process =   \\server\share\foo\rundll32.exe
//
//  Relative Path Example #3
//      pszCmdTemplate      =   regedit %1
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\windows\system32\regedit.exe
//              pszCommandLine  =   "regedit.exe" %1
//          * CreateProcess() would return TRUE
//              new process =   C:\Program Files\Example\regedit.bat
//
//  Relative Path Example #4
//      pszCmdTemplate      =   pbrush "%1"
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\windows\system32\mspaint.exe
//              pszCommandLine  =   "mspaint.exe" "%1"
//          * CreateProcess() would return FALSE
//
//  Relative Path Example #5
//      pszCmdTemplate      =   mycl "%1" "%2"
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Compilers\mycl.exe
//              pszCommandLine  =   "mycl.exe" "%1" "%2"
//          * CreateProcess() would return TRUE
//              new process =   C:\mycl.exe
//
//  Relative Path Example #6
//      pszCmdTemplate      =   myapp.exe
//          SHEvaluateSystemCommandTemplate() returns: CO_E_APPNOTFOUND
//          * CreateProcess() would return TRUE
//              new process =   \\server\share\foo\myapp.exe
//
//  Quoted Path Example #1
//      pszCmdTemplate      =   "C:\Program Files\Internet Explorer\iexplore.exe" -nohome
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet Explorer\iexplore.exe
//              pszCommandLine  =   "C:\Program Files\Internet Explorer\iexplore.exe" -nohome
//          CreateProcess() would return TRUE
//              new process =   C:\Program Files\Internet Explorer\iexplore.exe
//
//  Quoted Path Example #2
//      pszCmdTemplate      =   "C:\Program Files\Internet" -url
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet.exe
//              pszCommandLine  =   "C:\Program Files\Internet.exe" -url
//          CreateProcess() would return TRUE
//              new process =   C:\Program Files\internet.exe
//
//  Quoted Path Example #3
//      pszCmdTemplate      =   "C:\Program" -url
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program.exe
//              pszCommandLine  =   "C:\Program.exe" -url
//          CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #1
//      pszCmdTemplate      =   C:\Program Files\Internet Explorer\iexplore.exe -nohome
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet Explorer\iexplore.exe
//              pszCommandLine  =   "C:\Program Files\Internet Explorer\iexplore.exe" -nohome
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #2
//      pszCmdTemplate      =   C:\Program Files\Internet Explorer\iexplore.exe -url fool.htm
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet Explorer\iexplore.exe
//              pszCommandLine  =   "C:\Program Files\Internet Explorer\iexplore.exe" -url fool.htm
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #3
//      pszCmdTemplate      =   C:\Program Files\Internet Explorer\iexplore.exe -url C:\fool.htm
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet Explorer\iexplore.exe
//              pszCommandLine  =   "C:\Program Files\Internet Explorer\iexplore.exe" -url C:\fool.htm
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #4
//      pszCmdTemplate      =   C:\Program Files\Internet -url
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Program Files\Internet.exe
//              pszCommandLine  =   "C:\Program Files\Internet.exe" -url
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #5
//      pszCmdTemplate      =   C:\Other Programs\prog.exe -go %1 \fool %2
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Other Programs\prog.exe
//              pszCommandLine  =   "C:\Other Programs\prog.exe" %1 \fool %2
//          * CreateProcess() would return TRUE
//              new process =   C:\Other Programs\prog.exe
//
//  Unquoted Example #6
//      pszCmdTemplate      =   C:\Other Programs\prog.exe -go "\fool" "%1"
//          SHEvaluateSystemCommandTemplate() returns: S_OK
//              pszApplication  =   C:\Other Programs\prog.exe
//              pszCommandLine  =   "C:\Other Programs\prog.exe" -go "\fool" "%1"
//          * CreateProcess() would return TRUE
//              new process =   C:\Other Programs\prog.exe
//
//  Unquoted Example #7
//      pszCmdTemplate      =   C:\Program Files\Internet Explorer\iexplore.exe -url \fool.htm
//          SHEvaluateSystemCommandTemplate() returns: CO_E_APPNOTFOUND
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #8
//      pszCmdTemplate      =   C:\Program -url
//          SHEvaluateSystemCommandTemplate() returns: CO_E_APPNOTFOUND
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #9
//      pszCmdTemplate      =   C:\Other Programs\prog.exe -go \fool us
//          SHEvaluateSystemCommandTemplate() returns: CO_E_APPNOTFOUND
//          * CreateProcess() would return TRUE
//              new process =   C:\Other Programs\prog.exe
//
//  Unquoted Example #10
//      pszCmdTemplate      =   C:\Other Programs\prog.exe -go \fool %1
//          SHEvaluateSystemCommandTemplate() returns: CO_E_APPNOTFOUND
//          * CreateProcess() would return TRUE
//              new process =   C:\Other Programs\prog.exe
//
//  Unquoted Example #11
//      pszCmdTemplate      =   C:\Program "%1"
//          SHEvaluateSystemCommandTemplate() returns: E_ACCESSDENIED
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//
//  Unquoted Example #12
//      pszCmdTemplate      =   C:\Program
//          SHEvaluateSystemCommandTemplate() returns: E_ACCESSDENIED
//          * CreateProcess() would return TRUE
//              new process =   C:\Program.exe
//

//  used for implementing IShellFolder::GetUIObject(IID_IQueryAssociations)
//  designed for namespace extensions with registered extensible types
//  SHCreateDefaultContextMenu() and others use IQueryAssociations to build up data sets

typedef enum ASSOCCLASS
{                               //  which other members are used
    ASSOCCLASS_SHELL_KEY = 0,   //  hkeyClass
    ASSOCCLASS_PROGID_KEY,      //  hkeyClass
    ASSOCCLASS_PROGID_STR,      //  pszClass (HKCR\pszClass)
    ASSOCCLASS_CLSID_KEY,       //  hkeyClass
    ASSOCCLASS_CLSID_STR,       //  pszClass (HKCR\CLSID\pszClass)
    ASSOCCLASS_APP_KEY,         //  hkeyClass
    ASSOCCLASS_APP_STR,         //  pszClass (HKCR\Applications\PathFindFileName(pszClass))
    ASSOCCLASS_SYSTEM_STR,      //  pszClass
    ASSOCCLASS_FOLDER,          //  none
    ASSOCCLASS_STAR,            //  none
} ASSOCCLASS;
typedef struct ASSOCIATIONELEMENT_
{
    ASSOCCLASS ac;              // required
    HKEY hkClass;               // may be NULL
    PCWSTR pszClass;            // may be NULL
} ASSOCIATIONELEMENT;

// the object returned from this API implements IQueryAssociations

SHSTDAPI AssocCreateForClasses(__in const ASSOCIATIONELEMENT *rgClasses, ULONG cClasses, REFIID riid, __out void **ppv);

/* #if 0 // SAMPLE CODE
HRESULT CCustomFolder::_AssocCreate(PCUITEMID_CHILD pidl, REFIID riid, __out void **ppv)
{
    *ppv = NULL;
    ASSOCIATIONELEMENT rgAssoc[] =
    {
        { ASSOCCLASS_PROGID_STR, NULL, CCustomFolder::_MapChildToType(pidl)},
        { ASSOCCLASS_FOLDER, NULL, NULL},
    };
    if (CCustomFolder::_IsFolder(pidl))
    {
        return AssocCreateForClasses(rgAssoc, ARRAYSIZE(rgAssoc), riid, ppv);
    }
    else
    {
        //  skip FOLDER at the end
        return AssocCreateForClasses(rgAssoc, ARRAYSIZE(rgAssoc)-1, riid, ppv);
    }
}

HRESULT CCustomFolder::GetUIObjectOf(...)
{
    //  validate parameters
    if (riid == IID_IQueryAssociations)
    {
        hr = _AssocCreate(apidl[0], riid, ppv);
    }
    //  else if ...
}
#endif // SAMPLE CODE
*/

#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

////
////  End ShellExecuteEx and family
////

#if (NTDDI_VERSION >= NTDDI_WIN2K)
//
// RecycleBin
//

// struct for query recycle bin info
typedef struct _SHQUERYRBINFO {
    DWORD   cbSize;
#if !defined(_MAC) || defined(_MAC_INT_64)
    __int64 i64Size;
    __int64 i64NumItems;
#else
    DWORDLONG i64Size;
    DWORDLONG i64NumItems;
#endif
} SHQUERYRBINFO, *LPSHQUERYRBINFO;


// flags for SHEmptyRecycleBin
//
#define SHERB_NOCONFIRMATION    0x00000001
#define SHERB_NOPROGRESSUI      0x00000002
#define SHERB_NOSOUND           0x00000004


SHSTDAPI SHQueryRecycleBinA(LPCSTR pszRootPath, __inout LPSHQUERYRBINFO pSHQueryRBInfo);
SHSTDAPI SHQueryRecycleBinW(LPCWSTR pszRootPath, __inout LPSHQUERYRBINFO pSHQueryRBInfo);
#ifdef UNICODE
#define SHQueryRecycleBin  SHQueryRecycleBinW
#else
#define SHQueryRecycleBin  SHQueryRecycleBinA
#endif // !UNICODE
SHSTDAPI SHEmptyRecycleBinA(HWND hwnd, LPCSTR pszRootPath, DWORD dwFlags);
SHSTDAPI SHEmptyRecycleBinW(HWND hwnd, LPCWSTR pszRootPath, DWORD dwFlags);
#ifdef UNICODE
#define SHEmptyRecycleBin  SHEmptyRecycleBinW
#else
#define SHEmptyRecycleBin  SHEmptyRecycleBinA
#endif // !UNICODE

////
//// end of RecycleBin
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)


////
//// Taskbar notification definitions
////

#if (NTDDI_VERSION >= NTDDI_LONGHORN)
typedef enum
{
    QUNS_NOT_PRESENT               = 1,    // The user is not present.  Heuristic check for modes like: screen saver, locked machine, non-active FUS session
    QUNS_BUSY                      = 2,    // The user is busy.  Heuristic check for modes like: full-screen app
    QUNS_RUNNING_D3D_FULL_SCREEN   = 3,    // full-screen (exlusive-mode) D3D app
    QUNS_PRESENTATION_MODE         = 4,    // Windows presentation mode (laptop feature) is turned on
    QUNS_ACCEPTS_NOTIFICATIONS     = 5,    // none of the above
} QUERY_USER_NOTIFICATION_STATE;

SHSTDAPI SHQueryUserNotificationState(__out QUERY_USER_NOTIFICATION_STATE *pquns);
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)


typedef struct _NOTIFYICONDATAA {
    DWORD cbSize;
    HWND hWnd;
    UINT uID;
    UINT uFlags;
    UINT uCallbackMessage;
    HICON hIcon;
#if (NTDDI_VERSION < NTDDI_WIN2K)
    CHAR   szTip[64];
#endif
#if (NTDDI_VERSION >= NTDDI_WIN2K)
    CHAR   szTip[128];
    DWORD dwState;
    DWORD dwStateMask;
    CHAR   szInfo[256];
    union {
        UINT  uTimeout;
        UINT  uVersion;  // used with NIM_SETVERSION, values 0, 3 and 4
    } DUMMYUNIONNAME;
    CHAR   szInfoTitle[64];
    DWORD dwInfoFlags;
#endif
#if (NTDDI_VERSION >= NTDDI_WINXP)
    GUID guidItem;
#endif
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
    HICON hBalloonIcon;
#endif
} NOTIFYICONDATAA, *PNOTIFYICONDATAA;
typedef struct _NOTIFYICONDATAW {
    DWORD cbSize;
    HWND hWnd;
    UINT uID;
    UINT uFlags;
    UINT uCallbackMessage;
    HICON hIcon;
#if (NTDDI_VERSION < NTDDI_WIN2K)
    WCHAR  szTip[64];
#endif
#if (NTDDI_VERSION >= NTDDI_WIN2K)
    WCHAR  szTip[128];
    DWORD dwState;
    DWORD dwStateMask;
    WCHAR  szInfo[256];
    union {
        UINT  uTimeout;
        UINT  uVersion;  // used with NIM_SETVERSION, values 0, 3 and 4
    } DUMMYUNIONNAME;
    WCHAR  szInfoTitle[64];
    DWORD dwInfoFlags;
#endif
#if (NTDDI_VERSION >= NTDDI_WINXP)
    GUID guidItem;
#endif
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
    HICON hBalloonIcon;
#endif
} NOTIFYICONDATAW, *PNOTIFYICONDATAW;
#ifdef UNICODE
typedef NOTIFYICONDATAW NOTIFYICONDATA;
typedef PNOTIFYICONDATAW PNOTIFYICONDATA;
#else
typedef NOTIFYICONDATAA NOTIFYICONDATA;
typedef PNOTIFYICONDATAA PNOTIFYICONDATA;
#endif // UNICODE


#define NOTIFYICONDATAA_V1_SIZE     FIELD_OFFSET(NOTIFYICONDATAA, szTip[64])
#define NOTIFYICONDATAW_V1_SIZE     FIELD_OFFSET(NOTIFYICONDATAW, szTip[64])
#ifdef UNICODE
#define NOTIFYICONDATA_V1_SIZE      NOTIFYICONDATAW_V1_SIZE
#else
#define NOTIFYICONDATA_V1_SIZE      NOTIFYICONDATAA_V1_SIZE
#endif

#define NOTIFYICONDATAA_V2_SIZE     FIELD_OFFSET(NOTIFYICONDATAA, guidItem)
#define NOTIFYICONDATAW_V2_SIZE     FIELD_OFFSET(NOTIFYICONDATAW, guidItem)
#ifdef UNICODE
#define NOTIFYICONDATA_V2_SIZE      NOTIFYICONDATAW_V2_SIZE
#else
#define NOTIFYICONDATA_V2_SIZE      NOTIFYICONDATAA_V2_SIZE
#endif

#define NOTIFYICONDATAA_V3_SIZE     FIELD_OFFSET(NOTIFYICONDATAA, hBalloonIcon)
#define NOTIFYICONDATAW_V3_SIZE     FIELD_OFFSET(NOTIFYICONDATAW, hBalloonIcon)
#ifdef UNICODE
#define NOTIFYICONDATA_V3_SIZE      NOTIFYICONDATAW_V3_SIZE
#else
#define NOTIFYICONDATA_V3_SIZE      NOTIFYICONDATAA_V3_SIZE
#endif


#if (_WIN32_IE >= 0x0500)
#define NIN_SELECT          (WM_USER + 0)
#define NINF_KEY            0x1
#define NIN_KEYSELECT       (NIN_SELECT | NINF_KEY)
#endif

#if (_WIN32_IE >= 0x0501)
#define NIN_BALLOONSHOW         (WM_USER + 2)
#define NIN_BALLOONHIDE         (WM_USER + 3)
#define NIN_BALLOONTIMEOUT      (WM_USER + 4)
#define NIN_BALLOONUSERCLICK    (WM_USER + 5)
#endif
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#define NIN_POPUPOPEN           (WM_USER + 6)
#define NIN_POPUPCLOSE          (WM_USER + 7)
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)


#define NIM_ADD         0x00000000
#define NIM_MODIFY      0x00000001
#define NIM_DELETE      0x00000002
#if (_WIN32_IE >= 0x0500)
#define NIM_SETFOCUS    0x00000003
#define NIM_SETVERSION  0x00000004
// set NOTIFYICONDATA.uVersion with 0, 3 or 4
// please read the documentation on the behavior difference that the different versions imply
#define     NOTIFYICON_VERSION      3
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#define     NOTIFYICON_VERSION_4    4
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)
#endif

#define NIF_MESSAGE     0x00000001
#define NIF_ICON        0x00000002
#define NIF_TIP         0x00000004
#if (_WIN32_IE >= 0x0500)
#define NIF_STATE       0x00000008
#define NIF_INFO        0x00000010
#endif
#if (_WIN32_IE >= 0x600)
#define NIF_GUID        0x00000020
#endif
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#define NIF_REALTIME    0x00000040
#define NIF_SHOWTIP     0x00000080
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

#if (_WIN32_IE >= 0x0500)
#define NIS_HIDDEN              0x00000001
#define NIS_SHAREDICON          0x00000002

// says this is the source of a shared icon

// Notify Icon Infotip flags
#define NIIF_NONE       0x00000000
// icon flags are mutually exclusive
// and take only the lowest 2 bits
#define NIIF_INFO       0x00000001
#define NIIF_WARNING    0x00000002
#define NIIF_ERROR      0x00000003
#if (NTDDI_VERSION >= NTDDI_XPSP2) // also available in NTDDI_WS03SP1
#define NIIF_USER       0x00000004
#endif // (NTDDI_VERSION >= NTDDI_XPSP2)
#define NIIF_ICON_MASK  0x0000000F
#if (_WIN32_IE >= 0x0501)
#define NIIF_NOSOUND    0x00000010
#endif
#endif
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#define NIIF_LARGE_ICON 0x00000020
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

SHSTDAPI_(BOOL) Shell_NotifyIconA(DWORD dwMessage, __in PNOTIFYICONDATAA lpData);
SHSTDAPI_(BOOL) Shell_NotifyIconW(DWORD dwMessage, __in PNOTIFYICONDATAW lpData);
#ifdef UNICODE
#define Shell_NotifyIcon  Shell_NotifyIconW
#else
#define Shell_NotifyIcon  Shell_NotifyIconA
#endif // !UNICODE

////
//// End Taskbar Notification Icons
////

#ifndef SHFILEINFO_DEFINED
#define SHFILEINFO_DEFINED
////
//// Begin SHGetFileInfo
////

/*
 * The SHGetFileInfo API provides an easy way to get attributes
 * for a file given a pathname.
 *
 *   PARAMETERS
 *
 *     pszPath              file name to get info about
 *     dwFileAttributes     file attribs, only used with SHGFI_USEFILEATTRIBUTES
 *     psfi                 place to return file info
 *     cbFileInfo           size of structure
 *     uFlags               flags
 *
 *   RETURN
 *     TRUE if things worked
 */

typedef struct _SHFILEINFOA
{
        HICON       hIcon;                      // out: icon
        int         iIcon;                      // out: icon index
        DWORD       dwAttributes;               // out: SFGAO_ flags
        CHAR        szDisplayName[MAX_PATH];    // out: display name (or path)
        CHAR        szTypeName[80];             // out: type name
} SHFILEINFOA;
typedef struct _SHFILEINFOW
{
        HICON       hIcon;                      // out: icon
        int         iIcon;                      // out: icon index
        DWORD       dwAttributes;               // out: SFGAO_ flags
        WCHAR       szDisplayName[MAX_PATH];    // out: display name (or path)
        WCHAR       szTypeName[80];             // out: type name
} SHFILEINFOW;
#ifdef UNICODE
typedef SHFILEINFOW SHFILEINFO;
#else
typedef SHFILEINFOA SHFILEINFO;
#endif // UNICODE


// NOTE: This is also in shlwapi.h.  Please keep in synch.
#endif // !SHFILEINFO_DEFINED

#define SHGFI_ICON              0x000000100     // get icon
#define SHGFI_DISPLAYNAME       0x000000200     // get display name
#define SHGFI_TYPENAME          0x000000400     // get type name
#define SHGFI_ATTRIBUTES        0x000000800     // get attributes
#define SHGFI_ICONLOCATION      0x000001000     // get icon location
#define SHGFI_EXETYPE           0x000002000     // return exe type
#define SHGFI_SYSICONINDEX      0x000004000     // get system icon index
#define SHGFI_LINKOVERLAY       0x000008000     // put a link overlay on icon
#define SHGFI_SELECTED          0x000010000     // show icon in selected state
#if (NTDDI_VERSION >= NTDDI_WIN2K)
#define SHGFI_ATTR_SPECIFIED    0x000020000     // get only specified attributes
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)
#define SHGFI_LARGEICON         0x000000000     // get large icon
#define SHGFI_SMALLICON         0x000000001     // get small icon
#define SHGFI_OPENICON          0x000000002     // get open icon
#define SHGFI_SHELLICONSIZE     0x000000004     // get shell size icon
#define SHGFI_PIDL              0x000000008     // pszPath is a pidl
#define SHGFI_USEFILEATTRIBUTES 0x000000010     // use passed dwFileAttribute

#if (_WIN32_IE >= 0x0500)
#define SHGFI_ADDOVERLAYS       0x000000020     // apply the appropriate overlays
#define SHGFI_OVERLAYINDEX      0x000000040     // Get the index of the overlay
                                                // in the upper 8 bits of the iIcon
#endif

SHSTDAPI_(DWORD_PTR) SHGetFileInfoA(LPCSTR pszPath, DWORD dwFileAttributes, __inout_bcount_opt(cbFileInfo) SHFILEINFOA *psfi,
    UINT cbFileInfo, UINT uFlags);
SHSTDAPI_(DWORD_PTR) SHGetFileInfoW(LPCWSTR pszPath, DWORD dwFileAttributes, __inout_bcount_opt(cbFileInfo) SHFILEINFOW *psfi,
    UINT cbFileInfo, UINT uFlags);
#ifdef UNICODE
#define SHGetFileInfo  SHGetFileInfoW
#else
#define SHGetFileInfo  SHGetFileInfoA
#endif // !UNICODE

#if (NTDDI_VERSION >= NTDDI_LONGHORN)
typedef struct _SHSTOCKICONINFO
{
    DWORD cbSize;
    HICON hIcon;
    int   iSysImageIndex;
    int   iIcon;
    WCHAR szPath[MAX_PATH];
} SHSTOCKICONINFO;

#define SHGSI_ICONLOCATION      0 // you always get the icon location
#define SHGSI_ICON              SHGFI_ICON
#define SHGSI_SYSICONINDEX      SHGFI_SYSICONINDEX
#define SHGSI_LINKOVERLAY       SHGFI_LINKOVERLAY
#define SHGSI_SELECTED          SHGFI_SELECTED
#define SHGSI_LARGEICON         SHGFI_LARGEICON
#define SHGSI_SMALLICON         SHGFI_SMALLICON
#define SHGSI_SHELLICONSIZE     SHGFI_SHELLICONSIZE

//  Shell icons


typedef enum SHSTOCKICONID
{
    SIID_DOCNOASSOC = 0,          // document (blank page), no associated program
    SIID_DOCASSOC = 1,            // document with an associated program
    SIID_APPLICATION = 2,         // generic application with no custom icon
    SIID_FOLDER = 3,              // folder (closed)
    SIID_FOLDEROPEN = 4,          // folder (open)
    SIID_DRIVE525 = 5,            // 5.25" floppy disk drive
    SIID_DRIVE35 = 6,             // 3.5" floppy disk drive
    SIID_DRIVEREMOVE = 7,         // removable drive
    SIID_DRIVEFIXED = 8,          // fixed (hard disk) drive
    SIID_DRIVENET = 9,            // network drive
    SIID_DRIVENETDISABLED = 10,   // disconnected network drive
    SIID_DRIVECD = 11,            // CD drive
    SIID_DRIVERAM = 12,           // RAM disk drive
    SIID_WORLD = 13,              // entire network
    SIID_SERVER = 15,             // a computer on the network
    SIID_PRINTER = 16,            // printer
    SIID_MYNETWORK = 17,          // My network places
    SIID_FIND = 22,               // Find
    SIID_HELP = 23,               // Help
    SIID_SHARE = 28,              // overlay for shared items
    SIID_LINK = 29,               // overlay for shortcuts to items
    SIID_SLOWFILE = 30,           // overlay for slow items
    SIID_RECYCLER = 31,           // empty recycle bin
    SIID_RECYCLERFULL = 32,       // full recycle bin
    SIID_MEDIACDAUDIO = 40,       // Audio CD Media
    SIID_LOCK = 47,               // Security lock
    SIID_AUTOLIST = 49,           // AutoList
    SIID_PRINTERNET = 50,         // Network printer
    SIID_SERVERSHARE = 51,        // Server share
    SIID_PRINTERFAX = 52,         // Fax printer 
    SIID_PRINTERFAXNET = 53,      // Networked Fax Printer
    SIID_PRINTERFILE = 54,        // Print to File 
    SIID_STACK = 55,              // Stack
    SIID_MEDIASVCD = 56,          // SVCD Media
    SIID_STUFFEDFOLDER = 57,      // Folder containing other items
    SIID_DRIVEUNKNOWN = 58,       // Unknown drive
    SIID_DRIVEDVD = 59,           // DVD Drive
    SIID_MEDIADVD = 60,           // DVD Media
    SIID_MEDIADVDRAM = 61,        // DVD-RAM Media
    SIID_MEDIADVDRW = 62,         // DVD-RW Media
    SIID_MEDIADVDR = 63,          // DVD-R Media
    SIID_MEDIADVDROM = 64,        // DVD-ROM Media
    SIID_MEDIACDAUDIOPLUS = 65,   // CD+ (Enhanced CD) Media 
    SIID_MEDIACDRW = 66,          // CD-RW Media 
    SIID_MEDIACDR = 67,           // CD-R Media
    SIID_MEDIACDBURN = 68,        // Burning CD
    SIID_MEDIABLANKCD = 69,       // Blank CD Media 
    SIID_MEDIACDROM = 70,         // CD-ROM Media
    SIID_AUDIOFILES = 71,         // Audio files
    SIID_IMAGEFILES = 72,         // Image files
    SIID_VIDEOFILES = 73,         // Video files
    SIID_MIXEDFILES = 74,         // Mixed files
    SIID_FOLDERBACK = 75,         // Folder back 
    SIID_FOLDERFRONT = 76,        // Folder front
    SIID_SHIELD = 77,             // Security shield. Use for UAC prompts only.
    SIID_WARNING = 78,            // Warning
    SIID_INFO = 79,               // Informational
    SIID_ERROR = 80,              // Error
    SIID_KEY = 81,                // Key / Secure
    SIID_SOFTWARE = 82,           // Software
    SIID_RENAME = 83,             // Rename
    SIID_DELETE = 84,             // Delete
    SIID_MEDIAAUDIODVD = 85,      // Audio DVD Media
    SIID_MEDIAMOVIEDVD = 86,      // Movie DVD Media
    SIID_MEDIAENHANCEDCD = 87,    // Enhanced CD Media
    SIID_MEDIAENHANCEDDVD = 88,   // Enhanced DVD Media
    SIID_MEDIAHDDVD = 89,         // HD-DVD Media
    SIID_MEDIABLURAY = 90,        // BluRay Media
    SIID_MEDIAVCD = 91,           // VCD Media
    SIID_MEDIADVDPLUSR = 92,      // DVD+R Media
    SIID_MEDIADVDPLUSRW = 93,     // DVD+RW Media
    SIID_DESKTOPPC = 94,          // desktop computer
    SIID_MOBILEPC = 95,           // mobile computer (laptop/notebook)
    SIID_USERS = 96,              // users
    SIID_MEDIASMARTMEDIA = 97,    // Smart Media
    SIID_MEDIACOMPACTFLASH = 98,  // Compact Flash
    SIID_DEVICECELLPHONE = 99,    // Cell phone
    SIID_DEVICECAMERA = 100,      // Camera
    SIID_DEVICEVIDEOCAMERA = 101, // Video camera
    SIID_DEVICEAUDIOPLAYER = 102, // Audio player
    SIID_NETWORKCONNECT = 103,    // Connect to network
    SIID_INTERNET = 104,          // Internet
    SIID_ZIPFILE = 105,           // ZIP file
    SIID_SETTINGS = 106,          // Settings
    SIID_MAX_ICONS = 107,
} SHSTOCKICONID;

#define SIID_INVALID ((SHSTOCKICONID)-1)


SHSTDAPI SHGetStockIconInfo(SHSTOCKICONID siid, UINT uFlags, __inout SHSTOCKICONINFO *psii);

#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

#if (NTDDI_VERSION >= NTDDI_WIN2K)
#define SHGetDiskFreeSpace SHGetDiskFreeSpaceEx

SHSTDAPI_(BOOL) SHGetDiskFreeSpaceExA(LPCSTR pszDirectoryName, __out_opt ULARGE_INTEGER* pulFreeBytesAvailableToCaller,
    __out_opt ULARGE_INTEGER* pulTotalNumberOfBytes, __out_opt ULARGE_INTEGER* pulTotalNumberOfFreeBytes);
SHSTDAPI_(BOOL) SHGetDiskFreeSpaceExW(LPCWSTR pszDirectoryName, __out_opt ULARGE_INTEGER* pulFreeBytesAvailableToCaller,
    __out_opt ULARGE_INTEGER* pulTotalNumberOfBytes, __out_opt ULARGE_INTEGER* pulTotalNumberOfFreeBytes);
#ifdef UNICODE
#define SHGetDiskFreeSpaceEx  SHGetDiskFreeSpaceExW
#else
#define SHGetDiskFreeSpaceEx  SHGetDiskFreeSpaceExA
#endif // !UNICODE
SHSTDAPI_(BOOL) SHGetNewLinkInfoA(__in LPCSTR pszLinkTo, __in LPCSTR pszDir, __out_ecount(MAX_PATH) LPSTR pszName, __out BOOL *pfMustCopy, __in UINT uFlags);
SHSTDAPI_(BOOL) SHGetNewLinkInfoW(__in LPCWSTR pszLinkTo, __in LPCWSTR pszDir, __out_ecount(MAX_PATH) LPWSTR pszName, __out BOOL *pfMustCopy, __in UINT uFlags);
#ifdef UNICODE
#define SHGetNewLinkInfo  SHGetNewLinkInfoW
#else
#define SHGetNewLinkInfo  SHGetNewLinkInfoA
#endif // !UNICODE

#define SHGNLI_PIDL             0x000000001     // pszLinkTo is a pidl
#define SHGNLI_PREFIXNAME       0x000000002     // Make name "Shortcut to xxx"
#define SHGNLI_NOUNIQUE         0x000000004     // don't do the unique name generation
#if (_WIN32_IE >= 0x0501)
#define SHGNLI_NOLNK            0x000000008     // don't add ".lnk" extension
#endif // _WIN2_IE >= 0x0501
#if (_WIN32_IE >= 0x0600)
#define SHGNLI_NOLOCNAME        0x000000010     // use non localized (parsing) name from the target
#endif
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)


////
//// End SHGetFileInfo
////

#if (NTDDI_VERSION >= NTDDI_WIN2K)
// Printer stuff
#define PRINTACTION_OPEN           0
#define PRINTACTION_PROPERTIES     1
#define PRINTACTION_NETINSTALL     2
#define PRINTACTION_NETINSTALLLINK 3
#define PRINTACTION_TESTPAGE       4
#define PRINTACTION_OPENNETPRN     5
#ifdef WINNT
#define PRINTACTION_DOCUMENTDEFAULTS 6
#define PRINTACTION_SERVERPROPERTIES 7
#endif

SHSTDAPI_(BOOL) SHInvokePrinterCommandA(HWND hwnd, UINT uAction, LPCSTR lpBuf1, LPCSTR lpBuf2, BOOL fModal);
SHSTDAPI_(BOOL) SHInvokePrinterCommandW(HWND hwnd, UINT uAction, LPCWSTR lpBuf1, LPCWSTR lpBuf2, BOOL fModal);
#ifdef UNICODE
#define SHInvokePrinterCommand  SHInvokePrinterCommandW
#else
#define SHInvokePrinterCommand  SHInvokePrinterCommandA
#endif // !UNICODE
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)

#if (NTDDI_VERSION >= NTDDI_LONGHORN)
typedef struct _OPEN_PRINTER_PROPS_INFOA
{
    DWORD       dwSize;
    LPSTR       pszSheetName;
    UINT        uSheetIndex;
    DWORD       dwFlags;
    BOOL        bModal;
} OPEN_PRINTER_PROPS_INFOA, *POPEN_PRINTER_PROPS_INFOA;
typedef struct _OPEN_PRINTER_PROPS_INFOW
{
    DWORD       dwSize;
    LPWSTR      pszSheetName;
    UINT        uSheetIndex;
    DWORD       dwFlags;
    BOOL        bModal;
} OPEN_PRINTER_PROPS_INFOW, *POPEN_PRINTER_PROPS_INFOW;
#ifdef UNICODE
typedef OPEN_PRINTER_PROPS_INFOW OPEN_PRINTER_PROPS_INFO;
typedef POPEN_PRINTER_PROPS_INFOW POPEN_PRINTER_PROPS_INFO;
#else
typedef OPEN_PRINTER_PROPS_INFOA OPEN_PRINTER_PROPS_INFO;
typedef POPEN_PRINTER_PROPS_INFOA POPEN_PRINTER_PROPS_INFO;
#endif // UNICODE
#define PRINT_PROP_FORCE_NAME   0x01
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)


#endif /* WINVER >= 0x0400 */

#if (_WIN32_WINNT >= 0x0500) || (_WIN32_WINDOWS >= 0x0500)

//
// The SHLoadNonloadedIconOverlayIdentifiers API causes the shell's
// icon overlay manager to load any registered icon overlay
// identifers that are not currently loaded.  This is useful if an
// overlay identifier did not load at shell startup but is needed
// and can be loaded at a later time.  Identifiers already loaded
// are not affected.  Overlay identifiers implement the
// IShellIconOverlayIdentifier interface.
//
// Returns:
//      S_OK
//
SHSTDAPI SHLoadNonloadedIconOverlayIdentifiers(void);

//
// The SHIsFileAvailableOffline API determines whether a file
// or folder is available for offline use.
//
// Parameters:
//     pwszPath             file name to get info about
//     pdwStatus            (optional) OFFLINE_STATUS_* flags returned here
//
// Returns:
//     S_OK                 File/directory is available offline, unless
//                            OFFLINE_STATUS_INCOMPLETE is returned.
//     E_INVALIDARG         Path is invalid, or not a net path
//     E_FAIL               File/directory is not available offline
//
// Notes:
//     OFFLINE_STATUS_INCOMPLETE is never returned for directories.
//     Both OFFLINE_STATUS_LOCAL and OFFLINE_STATUS_REMOTE may be returned,
//     indicating "open in both places." This is common when the server is online.
//
SHSTDAPI SHIsFileAvailableOffline(__in LPCWSTR pwszPath, __out_opt LPDWORD pdwStatus);

#define OFFLINE_STATUS_LOCAL        0x0001  // If open, it's open locally
#define OFFLINE_STATUS_REMOTE       0x0002  // If open, it's open remotely
#define OFFLINE_STATUS_INCOMPLETE   0x0004  // The local copy is currently imcomplete.
                                            // The file will not be available offline
                                            // until it has been synchronized.

#endif

#if (NTDDI_VERSION >= NTDDI_WINXP)
//  sets the specified path to use the string resource
//  as the UI instead of the file system name
SHSTDAPI SHSetLocalizedName(LPCWSTR pszPath, LPCWSTR pszResModule, int idsRes);
#endif // (NTDDI_VERSION >= NTDDI_WINXP)
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
//  sets the specified path to use the string resource
//  as the UI instead of the file system name
SHSTDAPI SHRemoveLocalizedName(LPCWSTR pszPath);
//  gets the string resource for the specified path
SHSTDAPI SHGetLocalizedName(__in LPCWSTR pszPath, __out_ecount(cch) LPWSTR pszResModule, UINT cch, __out int *pidsRes);
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)


//====== ShellMessageBox ================================================

// If lpcTitle is NULL, the title is taken from hWnd
// If lpcText is NULL, this is assumed to be an Out Of Memory message
// If the selector of lpcTitle or lpcText is NULL, the offset should be a
//     string resource ID
// The variable arguments must all be 32-bit values (even if fewer bits
//     are actually used)
// lpcText (or whatever string resource it causes to be loaded) should
//     be a formatting string similar to wsprintf except that only the
//     following formats are available:
//         %%              formats to a single '%'
//         %nn%s           the nn-th arg is a string which is inserted
//         %nn%ld          the nn-th arg is a DWORD, and formatted decimal
//         %nn%lx          the nn-th arg is a DWORD, and formatted hex
//     note that lengths are allowed on the %s, %ld, and %lx, just
//                         like wsprintf
//

#if !defined(_SHLWAPI_)
#define LWSTDAPIV_(type)  EXTERN_C DECLSPEC_IMPORT type STDAPIVCALLTYPE
#else
#define LWSTDAPIV_(type)  STDAPIV_(type)
#endif

LWSTDAPIV_(int) ShellMessageBoxA(
    __in_opt HINSTANCE hAppInst,
    __in_opt HWND hWnd,
    __in LPCSTR lpcText,
    __in_opt LPCSTR lpcTitle,
    __in UINT fuStyle, ...);
LWSTDAPIV_(int) ShellMessageBoxW(
    __in_opt HINSTANCE hAppInst,
    __in_opt HWND hWnd,
    __in LPCWSTR lpcText,
    __in_opt LPCWSTR lpcTitle,
    __in UINT fuStyle, ...);
#ifdef UNICODE
#define ShellMessageBox  ShellMessageBoxW
#else
#define ShellMessageBox  ShellMessageBoxA
#endif // !UNICODE

#if (NTDDI_VERSION >= NTDDI_WIN2K)
SHSTDAPI_(BOOL) IsLFNDriveA(LPCSTR pszPath);
SHSTDAPI_(BOOL) IsLFNDriveW(LPCWSTR pszPath);
#ifdef UNICODE
#define IsLFNDrive  IsLFNDriveW
#else
#define IsLFNDrive  IsLFNDriveA
#endif // !UNICODE
#endif // (NTDDI_VERSION >= NTDDI_WIN2K)


#if         _WIN32_IE >= 0x0600

STDAPI          SHEnumerateUnreadMailAccountsA(HKEY hKeyUser, DWORD dwIndex, __out_ecount(cchMailAddress) LPSTR pszMailAddress, int cchMailAddress);
STDAPI          SHEnumerateUnreadMailAccountsW(HKEY hKeyUser, DWORD dwIndex, __out_ecount(cchMailAddress) LPWSTR pszMailAddress, int cchMailAddress);
#ifdef UNICODE
#define SHEnumerateUnreadMailAccounts  SHEnumerateUnreadMailAccountsW
#else
#define SHEnumerateUnreadMailAccounts  SHEnumerateUnreadMailAccountsA
#endif // !UNICODE
STDAPI          SHGetUnreadMailCountA(HKEY hKeyUser, LPCSTR pszMailAddress, __out_opt DWORD *pdwCount, __out_opt FILETIME *pFileTime, __out_ecount_opt(cchShellExecuteCommand) LPSTR pszShellExecuteCommand, int cchShellExecuteCommand);
STDAPI          SHGetUnreadMailCountW(HKEY hKeyUser, LPCWSTR pszMailAddress, __out_opt DWORD *pdwCount, __out_opt FILETIME *pFileTime, __out_ecount_opt(cchShellExecuteCommand) LPWSTR pszShellExecuteCommand, int cchShellExecuteCommand);
#ifdef UNICODE
#define SHGetUnreadMailCount  SHGetUnreadMailCountW
#else
#define SHGetUnreadMailCount  SHGetUnreadMailCountA
#endif // !UNICODE
STDAPI          SHSetUnreadMailCountA(LPCSTR pszMailAddress, DWORD dwCount, LPCSTR pszShellExecuteCommand);
STDAPI          SHSetUnreadMailCountW(LPCWSTR pszMailAddress, DWORD dwCount, LPCWSTR pszShellExecuteCommand);
#ifdef UNICODE
#define SHSetUnreadMailCount  SHSetUnreadMailCountW
#else
#define SHSetUnreadMailCount  SHSetUnreadMailCountA
#endif // !UNICODE

#endif  /*  _WIN32_IE >= 0x0600     */

#if (_WIN32_IE >= 0x0601)
STDAPI_(BOOL)   SHTestTokenMembership(__in_opt HANDLE hToken, __in ULONG ulRID);
#endif // (_WIN32_IE >= 0x0601)

#if         _WIN32_IE >= 0x0600

#if (NTDDI_VERSION >= NTDDI_WINXP)
SHSTDAPI SHGetImageList(int iImageList, REFIID riid, __out void **ppvObj);
#endif // (NTDDI_VERSION >= NTDDI_WINXP)

#if (NTDDI_VERSION >= NTDDI_WINXP)
#define SHIL_LARGE          0   // normally 32x32
#define SHIL_SMALL          1   // normally 16x16
#define SHIL_EXTRALARGE     2
#define SHIL_SYSSMALL       3   // like SHIL_SMALL, but tracks system small icon metric correctly
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#define SHIL_JUMBO          4   // normally 256x256
#define SHIL_LAST           SHIL_JUMBO
#else
#define SHIL_LAST           SHIL_SYSSMALL
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)
#endif // (NTDDI_VERSION >= NTDDI_WINXP)


// Function call types for ntshrui folder sharing helpers
typedef HRESULT (STDMETHODCALLTYPE *PFNCANSHAREFOLDERW)(LPCWSTR pszPath);
typedef HRESULT (STDMETHODCALLTYPE *PFNSHOWSHAREFOLDERUIW)(HWND hwndParent, LPCWSTR pszPath);

#endif  /*  _WIN32_IE >= 0x0600     */

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#if !defined(_WIN64)
#include <poppack.h>
#endif

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
#pragma warning(pop)
#endif


#if (NTDDI_VERSION >= NTDDI_LONGHORN)
// API for new Network Address Control

// Instantiation
#define WC_NETADDRESS L"msctls_netaddress"
SHSTDAPI_(BOOL) InitNetworkAddressControl();

// Address Control Messages

// NCM_GETADDRESS returns the type of address that is present in the
// control (based on TBD Net Address flags).  If the input string has
// not been validated using this message will force the validation of
// the input string.  The WPARAM is a BOOL to determine to show the
// balloon tip.  The LPARAM is a pointer to the structure to fill in
// with the address type and address string.
#define NCM_GETADDRESS (WM_USER+1)
#define NetAddr_GetAddress(hwnd,pv) \
    (HRESULT)SNDMSG(hwnd,NCM_GETADDRESS,0,(LPARAM)pv)
typedef struct tagNC_ADDRESS
{
   struct NET_ADDRESS_INFO_ *pAddrInfo; // defined in iphlpapi.h
   USHORT  PortNumber;
   BYTE    PrefixLength;
} NC_ADDRESS, *PNC_ADDRESS;

// NCM_SETALLOWTYPE sets the type of addresses that the control will allow.
// The address flags are defined in iphlpapi.h
#define NCM_SETALLOWTYPE (WM_USER+2)
#define NetAddr_SetAllowType(hwnd,addrMask) \
    (HRESULT)SNDMSG(hwnd,NCM_SETALLOWTYPE,(WPARAM)addrMask,0)
// NCM_GETALLOWTYPE returns the currently allowed type mask.
#define NCM_GETALLOWTYPE (WM_USER+3)
#define NetAddr_GetAllowType(hwnd) \
    (DWORD)SNDMSG(hwnd,NCM_GETALLOWTYPE,0,0)

// NCM_DISPLAYERRORTIP displays the error balloon tip with the correct
// error string (based on the last failure from the NCM_GETADDRESS call
#define NCM_DISPLAYERRORTIP (WM_USER+4)
#define NetAddr_DisplayErrorTip(hwnd) \
    (HRESULT)SNDMSG(hwnd,NCM_DISPLAYERRORTIP,0,0)

#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

#if (NTDDI_VERSION >= NTDDI_LONGHORN)
// Returns the type of media (CD, DVD, Blank, etc) that is in the drive.
// dwMediaContent is set to a combination of ARCONTENT flags.
STDAPI SHGetDriveMedia(LPCWSTR pszDrive, __out DWORD *pdwMediaContent);
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

// CPL task apis
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

// Windows Parental Controls (WPC) query apis
#if (NTDDI_VERSION >= NTDDI_LONGHORN)
#endif // (NTDDI_VERSION >= NTDDI_LONGHORN)

#endif  /* _INC_SHELLAPI */


